<?php
// Employee Model

require_once 'config.php';

class Employee {

    private $db;

    public function __construct() {
        // Initialize database connection using constants from config.php
        $this->db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
        
        // Check for connection errors
        if ($this->db->connect_errno) {
            die('Database connection error: ' . $this->db->connect_error);
        }
    }
    // method to retrieve all employees from the database
    public function getAllEmployees() {
        $employees = array();

        // Perform database query to fetch employees
        $result = $this->db->query("SELECT * FROM employees");

        // Check if query was successful
        if ($result) {
            // Fetch associative array of employees
            while ($row = $result->fetch_assoc()) {
                $employees[] = $row;
            }

            // Free result set
            $result->free();
        } else {
            // Handle query error
            die('Database query error: ' . $this->db->error);
        }

        return $employees;
    }

     // Method to add employee data to the database
     public function addEmployee($employeeData, $educationData, $workExperienceData) {
        // Begin transaction
        $this->db->begin_transaction();

        // Insert into employees table
        $employeeInsertQuery = "INSERT INTO employees (Employee_ID, FirstName, MiddleName, LastName, Date_Hired, Department, Position, Status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $employeeStmt = $this->db->prepare($employeeInsertQuery);
        $employeeStmt->bind_param('ssssssss', $employeeData['employeeId'], $employeeData['firstName'], $employeeData['middleName'], $employeeData['lastName'], $employeeData['dateHired'], $employeeData['department'], $employeeData['position'], $employeeData['status']);
        $employeeSuccess = $employeeStmt->execute();
        $employeeStmt->close();

        if (!$employeeSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }

        // Insert into education table
        $educationInsertQuery = "INSERT INTO educational_background (Employee_ID, School, Year_Graduated, Course, Level) VALUES (?, ?, ?, ?, ?)";
        $educationStmt = $this->db->prepare($educationInsertQuery);
        $educationStmt->bind_param('sssss', $educationData['employeeId'], $educationData['school'], $educationData['yearGraduated'], $educationData['course'], $educationData['level']);
        $educationSuccess = $educationStmt->execute();
        $educationStmt->close();

        if (!$educationSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }

        // Insert into experience table for each work experience entry
        foreach ($workExperienceData as $experience) {
            // Concatenate fromDate and toDate for each experience entry
            $inclusiveDate = isset($experience['inclusiveDate']) ? $experience['inclusiveDate'] : null;
            
            $experienceInsertQuery = "INSERT INTO work_experience (Employee_ID, Company, Position, Salary, Inclusive_Date) VALUES (?, ?, ?, ?, ?)";
            $experienceStmt = $this->db->prepare($experienceInsertQuery);
            $experienceStmt->bind_param('sssss', $employeeData['employeeId'], $experience['company'], $experience['workPosition'], $experience['salary'], $inclusiveDate);
            $experienceSuccess = $experienceStmt->execute();
            $experienceStmt->close();

            if (!$experienceSuccess) {
                // Rollback transaction and return false
                $this->db->rollback();
                return false;
            }
        }

                // Commit transaction and return true
                $this->db->commit();
                return true;
            }

        // Method to delete an employee from the database
        public function deleteEmployee($employeeId) {
            // Begin transaction
            $this->db->begin_transaction();

        // Delete employee data from the employees table
        $employeeDeleteQuery = "DELETE FROM employees WHERE Employee_ID = ?";
        $employeeStmt = $this->db->prepare($employeeDeleteQuery);
        $employeeStmt->bind_param('s', $employeeId);
        $employeeSuccess = $employeeStmt->execute();
        $employeeStmt->close();

        if (!$employeeSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }

        // Delete employee data from the educational_background table
        $educationDeleteQuery = "DELETE FROM educational_background WHERE Employee_ID = ?";
        $educationStmt = $this->db->prepare($educationDeleteQuery);
        $educationStmt->bind_param('s', $employeeId);
        $educationSuccess = $educationStmt->execute();
        $educationStmt->close();

        if (!$educationSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }

        // Delete employee data from the work_experience table
        $experienceDeleteQuery = "DELETE FROM work_experience WHERE Employee_ID = ?";
        $experienceStmt = $this->db->prepare($experienceDeleteQuery);
        $experienceStmt->bind_param('s', $employeeId);
        $experienceSuccess = $experienceStmt->execute();
        $experienceStmt->close();

        if (!$experienceSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }

    // Commit transaction and return true
    $this->db->commit();
    return true;
}

public function getEmployeeById($employeeId) {
    // Initialize an empty array to store employee data
    $employeeData = array();

    try {
        // Prepare the SQL query to retrieve employee data along with educational background
        $query = "
            SELECT employees.*, educational_background.*
            FROM employees
            LEFT JOIN educational_background ON employees.Employee_ID = educational_background.Employee_ID
            WHERE employees.Employee_ID = ?
        ";

        // Prepare the statement
        $stmt = $this->db->prepare($query);
        
        // Bind parameters
        $stmt->bind_param('i', $employeeId);

        // Execute the statement
        $stmt->execute();

        // Get the result set
        $result = $stmt->get_result();

        // Check if any rows were returned
        if ($result->num_rows > 0) {
            // Fetch employee data as associative arrays
            while ($row = $result->fetch_assoc()) {
                $employeeData[] = $row;
            }
        }

        // Close the statement
        $stmt->close();
    } catch (Exception $e) {
        // Handle exceptions
        // For example, log the error or return a default response
        error_log('Error: ' . $e->getMessage());
        return $employeeData; // Return empty array or another default response
    }

    return $employeeData;
}

public function getEmployeeWorkExperienceById($employeeId) {
    // Initialize an empty array to store work experience data
    $workExperienceData = array();

    try {
        // Prepare the SQL query to retrieve work experience data
        $query = "
            SELECT *
            FROM work_experience
            WHERE Employee_ID = ?
        ";

        // Prepare the statement
        $stmt = $this->db->prepare($query);
        
        // Bind parameters
        $stmt->bind_param('i', $employeeId);

        // Execute the statement
        $stmt->execute();

        // Get the result set
        $result = $stmt->get_result();

        // Check if any rows were returned
        if ($result->num_rows > 0) {
            // Fetch work experience data as associative arrays
            while ($row = $result->fetch_assoc()) {
                $workExperienceData[] = $row;
            }
        }

        // Close the statement
        $stmt->close();
    } catch (Exception $e) {
        // Handle exceptions
        // For example, log the error or return a default response
        error_log('Error: ' . $e->getMessage());
        return $workExperienceData; // Return empty array or another default response
    }

    return $workExperienceData;
}

// Method to update employee data in the database
public function updateEmployee($employeeData, $educationData, $workExperienceData) {
    // Begin transaction
    $this->db->begin_transaction();

    // Update employees table
    $employeeUpdateQuery = "UPDATE employees SET FirstName=?, MiddleName=?, LastName=?, Date_Hired=?, Department=?, Position=?, Status=? WHERE Employee_ID=?";
    $employeeStmt = $this->db->prepare($employeeUpdateQuery);
    $employeeStmt->bind_param('sssssssi', $employeeData['firstName'], $employeeData['middleName'], $employeeData['lastName'], $employeeData['dateHired'], $employeeData['department'], $employeeData['position'], $employeeData['status'], $employeeData['employeeId']);
    $employeeSuccess = $employeeStmt->execute();
    $employeeStmt->close();

    if (!$employeeSuccess) {
        // Rollback transaction and return false
        $this->db->rollback();
        return false;
    }

    // Update education table
    $educationUpdateQuery = "UPDATE educational_background SET School=?, Year_Graduated=?, Course=?, Level=? WHERE Employee_ID=?";
    $educationStmt = $this->db->prepare($educationUpdateQuery);
    $educationStmt->bind_param('ssssi', $educationData['school'], $educationData['yearGraduated'], $educationData['course'], $educationData['level'], $educationData['employeeId']);
    $educationSuccess = $educationStmt->execute();
    $educationStmt->close();

    if (!$educationSuccess) {
        // Rollback transaction and return false
        $this->db->rollback();
        return false;
    }

    // Delete existing work experience data for the employee
    $deleteExperienceQuery = "DELETE FROM work_experience WHERE Employee_ID=?";
    $deleteStmt = $this->db->prepare($deleteExperienceQuery);
    $deleteStmt->bind_param('i', $employeeData['employeeId']);
    $deleteSuccess = $deleteStmt->execute();
    $deleteStmt->close();

    if (!$deleteSuccess) {
        // Rollback transaction and return false
        $this->db->rollback();
        return false;
    }

    // Insert new work experience data
    foreach ($workExperienceData as $experience) {
        // Concatenate fromDate and toDate for each experience entry
        $inclusiveDate = isset($experience['inclusiveDate']) ? $experience['inclusiveDate'] : null;
        
        $experienceInsertQuery = "INSERT INTO work_experience (Employee_ID, Company, Position, Salary, Inclusive_Date) VALUES (?, ?, ?, ?, ?)";
        $experienceStmt = $this->db->prepare($experienceInsertQuery);
        $experienceStmt->bind_param('sssss', $employeeData['employeeId'], $experience['company'], $experience['workPosition'], $experience['salary'], $inclusiveDate);
        $experienceSuccess = $experienceStmt->execute();
        $experienceStmt->close();

        if (!$experienceSuccess) {
            // Rollback transaction and return false
            $this->db->rollback();
            return false;
        }
    }

    // Commit transaction and return true
    $this->db->commit();
    return true;
}

            
            

    // Close database connection
    public function __destruct() {
        $this->db->close();
    }
}
?>